require "lisp-error"

$lisp_value_stack = []

def lisp_value_pop ()
  tmp=$lisp_value_stack.shift
  if tmp
    tmp
  else
    $lisp_nil
  end
end

def self_eval_p (x)
  not(x.is_a?(Cons) or x.is_a?(Lsymbol))
end

def find_binding (symbol,binding)
  if binding[0] == :toplevel
    nil
  else found = binding.assoc(symbol)
    if found
      found
    else
      find_binding(symbol,binding[0])
    end
  end
end

# binding = [{value-binds},{function-binds},{block-binds}]

def lookup_value(symbol,binding)
  found = find_binding(symbol,binding[0])
  if found
    found[1]
  else
    symbol.symbol_value
  end
end

def lookup_function(symbol,binding)
  found = find_binding(symbol,binding[1])
  if found
    found[1]
  else
    symbol.symbol_function
  end
end

def lookup_block(symbol,binding)
  found = find_binding(symbol,binding[2])
  if found
    found[1]
  else
    nil
  end
end

def makeup_bindings(s_v_list)
  mapcar(lambda{|x| [x.car,x.cdr]},s_v_list).to_array
end

def extend_binding(s_v_list,binding)
  [binding] + makeup_bindings(s_v_list)
end

def extend_vbinding(list,bindings)
  [extend_binding(list,bindings[0]),bindings[1],bindings[2]]
end

def extend_fbinding(list,bindings)
  [bindings[0],extend_binding(list,bindings[1]),bindings[2]]
end


def extend_bbinding(name,sym,bindings)
  [bindings[0],bindings[1],[bindings[2],[name,sym]]]
end

def eval_if (sexp,binding)
  lisp_eval(sexp.car,binding)
  flag = lisp_value_pop
  if not(flag == $lisp_nil)
    lisp_eval(sexp.cdr.car,binding)
  else
    lisp_eval(sexp.cdr.cdr.car,binding)
  end
end

def eval_progn (sexp,binding)
  $lisp_value_stack=[$lisp_nil]
  eval_progn_iter(sexp,binding)
end

def eval_progn_iter (sexp,binding)
  if sexp == $lisp_nil
    nil
  else
    lisp_eval(sexp.car,binding)
    eval_progn_iter(sexp.cdr,binding)
  end
end

def eval_let (sexp,binding)
  s_v_list = mapcar(lambda{|x| if x.car.constant
                                 error "Cannot bind #{x.car.inspect} -- it is constant\n"
                               else
                                lisp_eval(x.cdr.car,binding);Cons.new(x.car,lisp_value_pop)
                               end},sexp.car)
  eval_progn(sexp.cdr,extend_vbinding(s_v_list,binding))
end

def eval_smlet (sexp,binding)
  s_v_list = mapcar(lambda{|x| Cons.new(x.car,LSymbolmacro.new(x.cdr.car))},sexp.car)
  eval_progn(sexp.cdr,extend_vbinding(s_v_list,binding))
end

def eval_flet (sexp,binding)
  fun_binds = sexp.car
  names = mapcar(lambda{|x| x.car},fun_binds)
  funs = mapcar(lambda{|x| (eval_function(Cons.new(find_symbol("lambda","cl"),x.cdr),binding);lisp_value_pop)},fun_binds)
  s_v_list = mapcar(lambda{|x,y| Cons.new(x,y)},names,funs)
  eval_progn(sexp.cdr,extend_fbinding(s_v_list,binding))
end

def macrofunction_generator (arglist,body)
  separated=remove_environment(arglist)
  dest=separated[0]
  env=separated[1]
  macroargs=find_symbol("**macroargs**","sys")
  envarg=find_symbol("**env**","sys")
  list(find_symbol("lambda","cl"),list(macroargs,envarg),
       list(find_symbol("let","cl"),(if env==$lisp_nil
                                       $lisp_nil
                                     else
                                       list(list(env,envarg))
                                     end),
            append(list(find_symbol("let*","cl"),destruc_top(Cons.new(Lsymbol.new(nil,nil),dest),macroargs)),body)))
end

def eval_macrolet (sexp,binding)
  fun_binds = sexp.car
  names = mapcar(lambda{|x| x.car},fun_binds)
  funs = mapcar(lambda{|x| (eval_function(macrofunction_generator(x.cdr.car,x.cdr.cdr),binding);tmp=lisp_value_pop;tmp.assume_macro;tmp)},fun_binds)
  s_v_list = mapcar(lambda{|x,y| Cons.new(x,y)},names,funs)
  eval_progn(sexp.cdr,extend_fbinding(s_v_list,binding))
end

def labels_helper (s_v_list,binding)
  if s_v_list == $lisp_nil
    nil
  else
    now=s_v_list.car.cdr
    now.arglist=Lcompiledarglist.new(now.arglist,binding)
    dummy_bind = now.arglist.dummy_bind(binding)
    now.body=mapcar(lambda{|x| compile_body(x,dummy_bind)},now.body)
    labels_helper(s_v_list.cdr,binding)
  end
end

def eval_labels (sexp,binding)
  fun_binds = sexp.car
  tmp_s_v = mapcar(lambda{|x| Cons.new(x.car,LInterpretedfunction.new(nil,x.cdr.car,x.cdr.cdr))},fun_binds)
  next_binding = extend_fbinding(tmp_s_v,binding)
  labels_helper(tmp_s_v,next_binding)
  eval_progn(sexp.cdr,next_binding)
end

# eval_function is in "lisp-function.rb"

def macrolet_expand (symbol,binding)
  found = find_binding(symbol,binding[0])
  if found
    if found[1].is_a?(LSymbolmacro)
      found[1].body
    else
      symbol
    end
  else
    symbol
  end
end

def eval_setq (sexp,binding)
  if sexp==$lisp_nil
    nil
  else
    symbol = macrolet_expand(sexp.car,binding)
    lisp_eval(sexp.cdr.car,binding)
    eval_setq_iter(symbol,lisp_value_pop,binding)
    eval_setq(sexp.cdr.cdr,binding)
  end
end

def eval_setq_iter (symbol,value,binding)
  if symbol.is_a?(Lsymbol)
    found = find_binding(symbol,binding[0])
    $lisp_value_stack = [(if found
      found[1] = value
    else
      symbol.set_symbol_value(value)
    end)]
  elsif symbol.is_a?(Lcompiledargument)
    found = symbol.found_binding
    if symbol.type==:lexical
      found = symbol.found_binding
    elsif symbol.type==:dynamic
      found = get_address(symbol.address,binding[0])
    else
      found=find_binding(symbol.address,binding[0])
    end
    $lisp_value_stack = [(if found
      found[1] = value
    else
      symbol.address.set_symbol_value(value)
    end)]
  else
    error "Error -- setq receives non symbol argument #{symbol.inspect}\n"
  end
end

def eval_psetq (sexp,binding)
  symbols=[]
  values=[]
  loop{
    symbols.push(macrolet_expand(sexp.car,binding))
    values.push((lisp_eval(sexp.cdr.car,binding);lisp_value_pop))
    sexp=sexp.cdr.cdr
    if sexp == $lisp_nil
      break
    end}
  loop{
    symbol=symbols.shift
    value=values.shift
    if symbol
      eval_setq_iter(symbol,value,binding)
    else
      break
    end}
end

def eval_mvbind (sexp,binding)
  mvlist = sexp.car
  form = sexp.cdr.car
  body = sexp.cdr.cdr
  lisp_eval(form,binding)
  s_v_list = mapcar(lambda{|x| Cons.new(x,lisp_value_pop)},mvlist)
  eval_progn(body,extend_vbinding(s_v_list,binding))
end

def eval_igerror (sexp,binding)
  begin
    eval_progn(sexp,binding)
  rescue LsimpleError => e
    e.protected.each{|x| x.call}
    $lisp_value_stack = [$lisp_nil,e]
  end
end

def eval_uwprotect (sexp,binding)
  protected_form = sexp.car
  cleanup_form =sexp.cdr
  begin
    lisp_eval(protected_form,binding)
    tmp=lisp_value_pop
    eval_progn(cleanup_form,binding)
    $lisp_value_stack=[tmp]
  rescue LCondition => c
    uw_handler(c,cleanup_form,binding)
  end
end

def uw_handler (c,cleanup_form,binding)
  if c.is_a?(LsimpleError) # throw error to toplevel is may back to here, so delay eval cleanup_forms
    c.protected.push(lambda{eval_progn(cleanup_form,binding)})
    raise c
  else # defualt action is [eval cleanup_form -> throw condition]
    eval_progn(cleanup_form,binding)
    raise c
  end
end

def eval_quote (sexp,binding)
  $lisp_value_stack = [sexp.car]
end

def eval_block (sexp,binding)
  name=Lsymbol.new(nil,nil)
  new_bind=extend_bbinding(sexp.car,name,binding)
  begin
    eval_progn(sexp.cdr,new_bind)
  rescue Lblockcondition => e
    if e.name==name
      $lisp_value_stack = e.values
    else
      raise e
    end
  end
end

def eval_return_from (sexp,binding)
  sym_name = sexp.car
  name=(if sym_name.is_a?(Lcompiledargument)
          sym_name.found_block_name(binding)
        else
          lookup_block(sexp.car,binding)
        end)
  if name.nil?
    name=sym_name
  end
  lisp_eval(sexp.cdr.car,binding)
  raise Lblockcondition.new(name,$lisp_value_stack)
end

def eval_go (sexp,binding)
  raise Ltagcondition.new(sexp.car)
end

def eval_tagbody (sexp,binding)
  tag_and_forms=sexp.to_array
  pos=0
  catch(:end){
    loop{begin
           x=tag_and_forms[pos]
           if x.nil?
             throw :end
           elsif x.is_a?(Lsymbol)
             nil
             pos+=1
           else
             lisp_eval(x,binding)
             pos+=1
           end
         rescue Ltagcondition => c
           new_pos=tag_and_forms.find_pos{|x| x == c.tag}
           pos=new_pos
         end}}
  $lisp_value_stack = [$lisp_nil]
end

def eval_catch (sexp,binding)
  lisp_eval(sexp.car,binding)
  tag=lisp_value_pop
  begin
    eval_progn(sexp.cdr,binding)
  rescue Lthrowcondition => c
    if c.name==tag
      $lisp_value_stack = c.values
    else
      raise c
    end
  end
end

def eval_throw (sexp,binding)
  lisp_eval(sexp.car,binding)
  tag=lisp_value_pop
  lisp_eval(sexp.cdr.car,binding)
  raise Lthrowcondition.new(tag,$lisp_value_stack)
end

$top_level_environment = [[:toplevel],[:toplevel],[:toplevel]]

def defspecial_eval (name,ruby_name)
  eval "(find_symbol(\"#{name}\",\"cl\")).function.eval=lambda{|sexp,binding| #{ruby_name}(sexp,binding)}"
end

["quote",
"if",
"progn",
"let",
"setq",
"psetq",
"flet",
"macrolet",
"labels",
"block",
"tagbody",
"go",
"catch",
"throw"].each{|x| defspecial_eval(x,"eval_#{x}")}

defspecial_eval("symbol-macrolet","eval_smlet")
defspecial_eval("multiple-value-bind","eval_mvbind")
defspecial_eval("ignore-errors","eval_igerror")
defspecial_eval("unwind-protect","eval_uwprotect")
defspecial_eval("return-from","eval_return_from")

find_symbol("function","cl").function.eval=lambda{|sexp,binding| eval_function(sexp.car,binding)}

def lisp_eval (sexp,binding=$top_level_environment)
  if sexp.is_a?(Lcompiledargument)
    $lisp_value_stack = [sexp.found_value(binding)]
  elsif self_eval_p(sexp)
    $lisp_value_stack = [sexp]
  elsif sexp.is_a?(Lsymbol)
    tmp = lookup_value(sexp,binding)
    if tmp.is_a?(LSymbolmacro)
      tmp.call(binding)
    else
      $lisp_value_stack = [tmp]
    end
  else
    car = sexp.car
    eval_function(car,binding,true)
    func = lisp_value_pop
    if func.is_a?(Lspecial)
      func.call(sexp.cdr,binding)
    elsif is_macro?(func)
        apply(func,list(sexp,binding),binding)
        lisp_eval(lisp_value_pop,binding)
    else
      apply(func,mapcar(lambda{|x| lisp_eval(x,binding);lisp_value_pop},sexp.cdr),binding)
    end
  end
end

def apply(func,arglist,binding)
  if func.is_a?(LPrimitivefunction)
    func.call(binding,*arglist.to_array)
  else
    func.call(arglist,binding)
  end
end