require "util"
require "cons"
require "lisp-symbol"
require "lisp-stream"
require "lisp-function"
require "lisp-eval"
require "lisp-char"

$context = ""

def read_delimited_list(endchar, stream = Linputstream.new)
  top=stream.fast_read_top
  if top==endchar.code
    $lisp_nil
  elsif top==" "[0] or top=="\n"[0] or top=="\t"[0] or top=="\r"[0]
    stream.read 1
    read_delimited_list(endchar,stream)
  else-
    car = read1(stream)
    Cons.new(car,read_delimited_list(endchar,stream))
  end
end

def read_list (endchar,stream)
  $context = ")"+$context
  car = read1(stream)
  $context = $context[1,$context.size]
  if car.nil?
    $lisp_nil
  elsif car == :dot
    $context = ")"+$context
    last = read1(stream)
    $context = $context[1,$context.size]
    if stream.fast_read_top == endchar
      stream.read 1
      last
    else
      error "more than one object after dot\n"
    end
  else
    Cons.new(car,read_list(endchar,stream))
  end
end

def read1 (stream)
  top = stream.fast_read_top
  reader = get_reader(top)
  if reader.is_a?(LPrimitivefunction)
    stream.read 1
    reader.call($top_level_environment,stream,top)
    $lisp_value_stack.shift
  elsif reader.is_a?(LInterpretedfunction)
    stream.read 1
    reader.call(list(stream,top),$top_level_environment)
    $lisp_value_stack.shift
  else
    read_token(stream)
  end
end

$reader_function_table = {}

$dispatch_macro_function_table = {}

def def_reader_func (key,name,&proc)
  new = LPrimitivefunction.new(Lsymbol.new(name,"cl"),lambda{|*args| $lisp_value_stack = [proc.call(*args)]})
  find_symbol(name,"cl").set_symbol_function(new)
  $reader_function_table[key] = new
end

def get_reader (key)
  $reader_function_table[key]
end

def get_macro_character (char)
  $reader_function_table[char.code]
end

def set_macro_character (char, closure)
  $reader_function_table[char.code] = closure
end

def make_dispatch_macro_character (char)
  let = {}
  $dispatch_macro_function_table[char.code] = let
  def_reader_func(char.code,"dispatch-#{char.inspect}"){|stream,char|
    top=stream.fast_read_top
    reader = let[top]
    if reader
      stream.read 1
      reader.call(list(stream,char,top),$top_level_environment)
      $lisp_value_stack.shift
    else
      read1(stream)
    end}
end

def set_dispatch_macro_character (char1,char2,closure)
  table=$dispatch_macro_function_table[char1.code]
  if table
    table[char2.code] = closure
  else
     error "Error -- #{char1.inspect} is not a dipatch character\n"
  end
end

def_reader_func("("[0],"read-list"){|stream,char|
  read_list(")"[0],stream)}

def_reader_func(")"[0],"end-of-list"){|stream,char|
  if $context.find{|x| x == char}
    nil
  else
    error "Error -- ignoring extra right parenthesis [file position = #{stream.pos}]\n"
  end}

def_reader_func(" "[0],"read-space"){|stream,char|
  read1(stream)}

def_reader_func("'"[0],"read-quote"){|stream,char|
  list(find_symbol("quote","cl"),read1(stream))}

def_reader_func("\""[0],"read-string!"){|stream,char|
  read_string(char,stream)}

def_reader_func(";"[0],"read-comment"){|stream,char|
  loop{
    tmp = stream.read 1
    if (tmp == "\n" or tmp == "\r")
      break
    end}
  read1(stream)}

$reader_function_table["\n"[0]] = get_reader(" "[0])
$reader_function_table["\t"[0]] = get_reader(" "[0])
$reader_function_table["\r"[0]] = get_reader(" "[0])

def_reader_func("`"[0],"read-backquote"){|stream,char|
  $context = ","+$context
  tmp=read1(stream)
  $context = $context[1,$context.size]
  expand_comma(tmp)}

class Comma_object
  attr_accessor :object, :atp
  def initialize (object, atp = nil)
    @object = object; @atp= atp
  end
  def inspect
    (if @atp
      ",@"
     else
      ","
     end) + object.inspect
  end
end

def_reader_func(","[0],"read-comma"){|stream,char|
  pos=$context.find_pos{|x| x == char}
  if pos
    tmp = $context
    $context = $context[0,pos]+$context[pos+1,$context.length]
    if stream.fast_read_top == "@"[0]
      stream.read 1
      new = Comma_object.new(read1(stream),true)
      $context = tmp
      new
    else
      new = Comma_object.new(read1(stream),nil)
      $context = tmp
      new
    end
  else
   error "Error -- comma appears not inside of backquote\n"
  end}

def comma? (x)
  if x.is_a?(Cons)
    comma?(x.car) or comma?(x.cdr)
  else
    x.is_a?(Comma_object)
  end
end

def comma_at? (x)
  if x.is_a?(Cons)
    (x.car.is_a?(Comma_object) and x.car.atp) or comma_at?(x.cdr)
  else
    x.is_a?(Comma_object) and x.atp
  end
end

def compile_comma (x)
  if x.is_a?(Cons)
    car = x.car
    if comma?(car)
      Cons.new(expand_comma(car),compile_comma(x.cdr))
    else
      Cons.new(list(find_symbol("quote","cl"),car),compile_comma(x.cdr))
    end
  elsif x == $lisp_nil
    $lisp_nil
  else
    expand_comma(x)
  end
end

def compile_comma_at (x)
  if x.is_a?(Cons)
    car = x.car
    if comma_at?(car)
      if car.is_a?(Cons)
        Cons.new(list(find_symbol("list","cl"),Cons.new(find_symbol("append","cl"),compile_comma_at(car))),compile_comma_at(x.cdr))
      else
        Cons.new(car.object,compile_comma_at(x.cdr))
      end
    elsif comma?(car)
      Cons.new(list(find_symbol("list","cl"),expand_comma(car)),compile_comma_at(x.cdr))
    else
      Cons.new(list(find_symbol("quote","cl"),list(car)),compile_comma_at(x.cdr))
    end
  elsif not(x == $lisp_nil)
    if comma_at?(x)
      error "Error -- undefined backquote syntax\n"
    elsif comma?(x)
      expand_comma(x)
    else
      x
    end
  else
    $lisp_nil
  end
end

def expand_comma (x)
  if comma?(x)
    if comma_at?(x)
      Cons.new(find_symbol("append","cl"),compile_comma_at(x))
    elsif x.is_a?(Cons)
      Cons.new(find_symbol("list","cl"),compile_comma(x))
    else
      x.object
    end
  else
    list(find_symbol("quote","cl"),x)
  end
end

def read_string (endchar, stream, escaped = nil)
  top = stream.fast_read_top
  if top == endchar
    if escaped
      top2 = stream.read 1
      top2 + read_string(endchar,stream)
    else
      stream.read 1
      ""
    end
  elsif top == "\\"[0]
    if escaped
      top2 = stream.read 1
      top2 + read_string(endchar,stream)
    else
      stream.read 1
      read_string(endchar,stream,true)
    end
  else
    top2 = stream.read 1
    top2 + read_string(endchar,stream)
  end
end    

def read_token (stream, acc = "")
  first = stream.read 1
  second = get_reader(stream.fast_read_top)
  if second
    convert_lisp_object(acc+first)
  else
    read_token(stream, acc + first)
  end
end 

def num_string? (x)
  48<=x and x<=57 or x == "."[0] or x == "/"[0]
end

def convert_lisp_object (string)
  colon = string.find_pos{|x| x == ":"[0]}
  if colon == 0
    read_intern(string[1,string.length],"keyword")
  elsif colon
    read_intern(string[colon+1,string.length],string[0,colon])
  elsif string[0]=="-"[0]
    sub=string[1,string.length]
    if sub.length==0
      read_intern(string)
    elsif sub.every{|x| num_string? x}
      count_dot = sub.count_if{|x| x == "."[0]}
      count_slash = sub.count_if{|x| x == "/"[0]}
      if count_dot == 1 and count_slash == 0
        if sub == "."
          read_intern(string)
        else    
          Float(string)
        end
      elsif count_dot == 0 and count_slash == 1
        if sub == "/"
          read_intern(string)
        else
          pos = sub.find_pos{|x| x == "/"[0]}
          if pos>0
            0-Rational(make_integer(sub[0,pos]), make_integer(sub[pos+1,sub.length]))
          else
            read_intern(string)
          end
        end
      elsif count_dot == 0 and count_slash == 0
        make_integer(string)
      else
        read_intern(string)
      end
    else
      read_intern(string)
    end
  elsif string.every{|x| num_string? x}
    count_dot = string.count_if{|x| x == "."[0]}
    count_slash = string.count_if{|x| x == "/"[0]}
    if count_dot == 1 and count_slash == 0
      if string == "."
        if $context.find{|x| x == ")"[0]}
          :dot
        else
          error "Dot context error\n"
        end
      else    
        Float(string)
      end
    elsif count_dot == 0 and count_slash == 1
      if string == "/"
        read_intern(string)
      else
        pos = sub.find_pos{|x| x == "/"[0]}
        if pos>0
          0-Rational(make_integer(sub[0,pos]), make_integer(sub[pos+1,sub.length]))
        else
          read_intern(string)
        end
     end
    elsif count_dot == 0 and count_slash == 0
      make_integer(string)
    else
      read_intern(string)
    end
  else
    read_intern(string)
  end
end

def make_integer (string)
  loop{
    if string[0] == "0"[0]
      l=string.length
      if l==1
        break
      else
        string=string[1,l]
      end
    else
      break
    end}
  Integer(string)
end

# main
def read (stream = nil)
  if stream and not(stream==$lisp_nil)
    stream
  else
    stream = Linputstream.new
  end
  read1(stream)
end

def read_char (stream = nil)
  if stream
    stream
  else
    stream = Linputstream.new
  end
  Lchar.new((stream.read 1)[0])
end